# 🔒 Sistema Completo de Controle e Monitoramento
## Point Blank Royal Launcher

Este guia mostra como implementar o sistema completo de controle remoto, monitoramento em tempo real e dashboard administrativo.

---

## 📋 O Que Você Terá

### ✅ Controle Total:
- Ver quem está usando o launcher em tempo real
- Ver todos os processos rodando no PC do usuário
- Banir/desbanir usuários remotamente
- Estatísticas completas (dia/semana/mês)
- Detecção automática de hacks
- IP tracking
- Hardware ID único por PC

### ✅ Dashboard Web:
- Interface administrativa completa
- Atualização automática a cada 10 segundos
- Ver usuários online agora
- Histórico de detecções
- Gerenciar bans
- Estatísticas em tempo real

### ✅ Launcher:
- Reporta abertura ao servidor
- Envia lista de processos a cada 60s
- Heartbeat a cada 30s (mostra online)
- Detecta ban remoto em 10s
- Mostra tela de suporte se banido

---

## 🚀 Instalação Passo a Passo

### 1️⃣ Configurar Banco de Dados

**a) Criar banco de dados:**
```sql
mysql -u root -p
```

**b) Executar script:**
```bash
mysql -u root -p < database.sql
```

Isso cria:
- ✅ Banco `pointblank_launcher`
- ✅ 8 tabelas
- ✅ Views para consultas rápidas
- ✅ Procedures de limpeza

### 2️⃣ Upload dos Arquivos PHP

**Estrutura no servidor:**
```
/public_html/
├── api.php                  ← API principal do launcher
├── dashboard_api.php        ← API do dashboard
├── dashboard.html           ← Painel administrativo
├── blacklist.php            ← Lista de processos bloqueados
├── slides.json              ← Imagens do slider
└── update.json              ← Patches disponíveis
```

**Configurar conexão nos PHPs:**
```php
$host = 'localhost';
$user = 'seu_usuario_mysql';
$pass = 'sua_senha_mysql';
$db = 'pointblank_launcher';
```

### 3️⃣ Atualizar Launcher Python

**a) Adicione no início do arquivo:**
```python
from monitoring_module import get_hwid, RemoteControl, EnhancedAntiHackMonitor
```

**b) Configure URLs (linha ~23):**
```python
API_URL = "https://pbroyal.com/api.php"
```

**c) No `__init__` do PBLauncher:**
```python
def __init__(self):
    super().__init__()
    # ... código existente ...
    
    # ADICIONE ESTAS LINHAS:
    self.hwid = get_hwid()
    self.remote_control = RemoteControl(self.hwid)
    self.remote_control.banned.connect(self.on_banned)
    self.remote_control.start()
```

**d) Substitua AntiHackMonitor:**
```python
# ANTES:
self.monitor = AntiHackMonitor(self.blacklist)

# DEPOIS:
self.monitor = EnhancedAntiHackMonitor(self.blacklist, self.remote_control)
```

**e) Adicione método on_banned:**
```python
def on_banned(self, ban_data):
    self.hide()
    
    msg = QMessageBox(self)
    msg.setIcon(QMessageBox.Critical)
    msg.setWindowTitle("Conta Bloqueada")
    msg.setText("Sua conta foi bloqueada!")
    
    reason = ban_data.get('reason', 'Violação dos termos')
    permanent = ban_data.get('permanent', False)
    
    if permanent:
        detail = f"Motivo: {reason}\n\nBan PERMANENTE"
    else:
        until = ban_data.get('banned_until', 'Indefinido')
        detail = f"Motivo: {reason}\n\nBloqueado até: {until}"
    
    msg.setDetailedText(detail)
    
    # Botão de suporte
    support_btn = msg.addButton("Abrir Suporte", QMessageBox.ActionRole)
    msg.addButton(QMessageBox.Close)
    
    msg.exec_()
    
    if msg.clickedButton() == support_btn:
        import webbrowser
        webbrowser.open(ban_data.get('support_url', 'https://pbroyal.com/suporte'))
    
    QApplication.quit()
```

**f) No quit_application:**
```python
def quit_application(self):
    # ... código existente ...
    
    # ADICIONE:
    if self.remote_control and self.remote_control.isRunning():
        self.remote_control.stop()
        self.remote_control.wait()
```

### 4️⃣ Recompilar Launcher

```bash
pyinstaller --onefile --windowed --name="PBRoyalLauncher" pb_launcher.py
```

---

## 🎯 Como Funciona

### Fluxo do Sistema:

1. **Usuário abre o launcher:**
   - Launcher gera HWID único
   - Envia para `api.php?action=check_user`
   - Se banido → mostra tela de bloqueio
   - Se permitido → continua

2. **Launcher inicia o jogo:**
   - Envia para `api.php?action=report_launch`
   - Registra sessão no banco
   - Inicia heartbeat (30 em 30s)

3. **Monitoramento contínuo:**
   - A cada 30s: envia heartbeat (mostra online)
   - A cada 60s: envia lista de processos
   - A cada 10s: verifica se foi banido
   - A cada 1s: verifica processos da blacklist

4. **Detecção de hack:**
   - Launcher detecta processo suspeito
   - Envia para `api.php?action=report_process`
   - API registra detecção
   - Se 3+ detecções em 1 hora → BAN AUTOMÁTICO
   - Launcher encerra o jogo

5. **Ban remoto:**
   - Admin bane via dashboard
   - Em até 10s o launcher detecta
   - Mostra tela de bloqueio
   - Oferece botão de suporte

---

## 📊 Dashboard Administrativo

### Acessar:
```
https://pbroyal.com/dashboard.html
```

### Funcionalidades:

#### 📈 Estatísticas:
- Total de usuários cadastrados
- Usuários online AGORA (tempo real)
- Aberturas hoje/semana/mês
- Total de bans ativos
- Hacks detectados hoje

#### 👥 Usuários Online:
- Lista em tempo real
- PC Name, Username, IP, HWID
- Última atividade (minutos atrás)
- Botão "Banir" (ban instantâneo)
- Botão "Processos" (ver o que está rodando)

#### ⚠️ Detecções Recentes:
- Últimas 24 horas
- Processo detectado em destaque
- Botão para banir direto

#### 🚫 Usuários Banidos:
- Lista de todos os bans ativos
- Motivo, data, tipo (permanente/temporário)
- Botão "Desbanir"

### Auto-Refresh:
- Dashboard atualiza sozinho a cada 10 segundos
- Não precisa dar F5

---

## 🔐 Hardware ID (HWID)

### Como é Gerado:
```python
HWID = SHA256(
    Nome do PC +
    Sistema Operacional +
    Versão do OS +
    Processador +
    MAC Address
)
```

### Características:
- ✅ Único por computador
- ✅ Não muda ao reinstalar Windows
- ✅ Detecta troca de hardware
- ✅ Impossível falsificar facilmente

---

## 📡 APIs Disponíveis

### API Principal (api.php):

| Endpoint | Método | Descrição |
|----------|--------|-----------|
| `?action=check_user` | POST | Verifica se pode usar |
| `?action=report_launch` | POST | Registra abertura |
| `?action=report_process` | POST | Reporta hack detectado |
| `?action=check_ban` | POST | Verifica ban |
| `?action=heartbeat` | POST | Sinal de vida |
| `?action=get_stats` | GET | Estatísticas gerais |

### API Dashboard (dashboard_api.php):

| Endpoint | Método | Descrição |
|----------|--------|-----------|
| `?action=get_stats` | GET | Estatísticas |
| `?action=get_online_users` | GET | Usuários online |
| `?action=get_detections` | GET | Detecções recentes |
| `?action=get_bans` | GET | Lista de bans |
| `?action=ban_user` | POST | Banir usuário |
| `?action=unban_user` | POST | Desbanir usuário |
| `?action=get_processes` | GET | Ver processos |

---

## 🛡️ Sistema de Ban

### Ban Automático:
- 3+ detecções de hack em 1 hora = BAN PERMANENTE
- Registra automaticamente no banco
- Launcher detecta em até 10 segundos

### Ban Manual (Dashboard):
- Admin clica em "Banir"
- Digite o motivo
- Escolhe: Permanente ou 7 dias
- Usuário é bloqueado instantaneamente

### Desban:
- Admin clica em "Desbanir"
- Remove do banco imediatamente
- Usuário pode usar novamente

### Tela de Bloqueio:
```
╔══════════════════════════════╗
║   🚫 CONTA BLOQUEADA         ║
╠══════════════════════════════╣
║ Motivo: Uso de CheatEngine  ║
║ Tipo: PERMANENTE             ║
╠══════════════════════════════╣
║ [Abrir Suporte]  [Fechar]   ║
╚══════════════════════════════╝
```

---

## 📊 Tabelas do Banco

### users
- Todos os usuários que já usaram
- HWID, IP, PC Name, Username
- Primeira vez, última vez
- Total de aberturas

### bans
- Todos os banimentos
- HWID, motivo, banido por
- Permanente ou temporário
- Data de desbloqueio

### launcher_sessions
- Cada vez que alguém abre o launcher
- HWID, IP, hora de início
- Último heartbeat
- Hora que fechou

### process_detections
- Cada hack detectado
- HWID, processo, IP
- Data/hora da detecção

### realtime_processes
- Lista de processos AGORA
- HWID, lista completa (JSON)
- Atualizado a cada 60s

### support_tickets
- Tickets de suporte dos usuários
- HWID, assunto, mensagem
- Status, resposta do admin

---

## 🔧 Manutenção

### Limpeza Automática:

Execute diariamente via cron:
```bash
0 3 * * * mysql -u root -p pointblank_launcher -e "CALL cleanup_old_data()"
```

Isso remove:
- Heartbeats antigos (>7 dias)
- Processos em tempo real antigos (>1 hora)
- Atualiza estatísticas diárias

### Backup:

```bash
# Backup diário
mysqldump -u root -p pointblank_launcher > backup_$(date +%Y%m%d).sql

# Backup semanal
0 0 * * 0 mysqldump -u root -p pointblank_launcher | gzip > backup_$(date +%Y%m%d).sql.gz
```

---

## 🎯 Dicas de Segurança

1. **Proteja o dashboard:**
   - Use `.htpasswd` no Apache
   - Ou adicione login PHP
   - Só admin pode acessar

2. **SSL/HTTPS:**
   - SEMPRE use HTTPS
   - Certificado Let's Encrypt grátis

3. **Rate Limiting:**
   - Limite requisições por IP
   - Evita DDoS na API

4. **Validação:**
   - Sempre valide HWID
   - Sanitize inputs SQL

5. **Logs:**
   - Registre ações dos admins
   - Tabela `admin_logs`

---

## 📞 Suporte

Se o usuário for banido, ele vê:
- Motivo do ban
- Se é permanente ou temporário
- Botão "Abrir Suporte"

Crie página de suporte:
```
https://pbroyal.com/suporte
```

Com formulário para:
- HWID (auto-preenchido)
- Motivo do contato
- Mensagem
- Screenshot (opcional)

---

## ✅ Checklist de Implementação

- [ ] Banco de dados criado
- [ ] Tabelas criadas (database.sql)
- [ ] api.php configurado e testado
- [ ] dashboard_api.php configurado
- [ ] dashboard.html acessível
- [ ] Launcher atualizado com monitoramento
- [ ] Launcher recompilado
- [ ] Testado ban/desban
- [ ] Testado detecção de hack
- [ ] Dashboard funcionando
- [ ] Página de suporte criada
- [ ] Backup configurado
- [ ] Limpeza automática (cron)

---

## 🎉 Pronto!

Agora você tem controle total sobre quem usa seu servidor!

- 👁️ Veja quem está online
- 📊 Estatísticas completas
- 🚫 Bana quem usar hack
- 📡 Monitora processos em tempo real
- 💻 Dashboard profissional

**Acesse:** `https://pbroyal.com/dashboard.html`
