"""
Módulo de Monitoramento e Controle Remoto
Point Blank Royal Launcher

Adicione este código ao launcher principal
"""

import uuid
import platform
import socket
import getpass

# ============================================
# CONFIGURAÇÕES DE MONITORAMENTO
# ============================================
API_URL = "https://pbroyal.com/api.php"
HEARTBEAT_INTERVAL = 30  # segundos - envia status a cada 30s
PROCESS_REPORT_INTERVAL = 60  # segundos - envia lista de processos
BAN_CHECK_INTERVAL = 10  # segundos - verifica se foi banido
# ============================================


def get_hwid():
    """Gera HWID único baseado no hardware do PC"""
    import hashlib
    
    # Combina várias informações do PC para criar ID único
    pc_info = []
    
    # Nome do PC
    pc_info.append(platform.node())
    
    # Sistema operacional
    pc_info.append(platform.system())
    pc_info.append(platform.release())
    
    # Processador
    pc_info.append(platform.processor())
    
    # MAC Address (mais confiável)
    mac = ':'.join(['{:02x}'.format((uuid.getnode() >> elements) & 0xff)
                    for elements in range(0, 2*6, 2)][::-1])
    pc_info.append(mac)
    
    # Gera hash único
    hwid_string = '|'.join(pc_info)
    hwid = hashlib.sha256(hwid_string.encode()).hexdigest()
    
    return hwid


def get_pc_info():
    """Retorna informações do PC"""
    return {
        'pc_name': platform.node(),
        'username': getpass.getuser(),
        'os': f"{platform.system()} {platform.release()}",
        'processor': platform.processor()
    }


class RemoteControl(QThread):
    """Thread para comunicação com servidor"""
    banned = pyqtSignal(dict)  # Emite quando detecta ban
    
    def __init__(self, hwid):
        super().__init__()
        self.hwid = hwid
        self.session_id = None
        self.running = True
        self.pc_info = get_pc_info()
        
    def run(self):
        # Verifica se pode usar o launcher
        if not self.check_user():
            return
        
        # Reporta abertura do launcher
        self.report_launch()
        
        # Loop de monitoramento
        heartbeat_counter = 0
        process_counter = 0
        ban_counter = 0
        
        while self.running:
            # Heartbeat (a cada 30s)
            if heartbeat_counter >= HEARTBEAT_INTERVAL:
                self.heartbeat()
                heartbeat_counter = 0
            
            # Reporta processos (a cada 60s)
            if process_counter >= PROCESS_REPORT_INTERVAL:
                self.report_processes()
                process_counter = 0
            
            # Verifica ban (a cada 10s)
            if ban_counter >= BAN_CHECK_INTERVAL:
                if self.check_ban_status():
                    return
                ban_counter = 0
            
            self.msleep(1000)
            heartbeat_counter += 1
            process_counter += 1
            ban_counter += 1
    
    def check_user(self):
        """Verifica se usuário pode usar o launcher"""
        try:
            response = requests.post(f"{API_URL}?action=check_user", data={
                'hwid': self.hwid,
                'pc_name': self.pc_info['pc_name'],
                'username': self.pc_info['username']
            }, timeout=10)
            
            if response.status_code == 200:
                data = response.json()
                
                if not data.get('allowed', False):
                    # Usuário banido
                    if data.get('banned', False):
                        self.banned.emit(data)
                    return False
                
                return True
        except:
            # Se falhar conexão, permite jogar (offline mode)
            return True
        
        return False
    
    def report_launch(self):
        """Reporta abertura do launcher"""
        try:
            response = requests.post(f"{API_URL}?action=report_launch", data={
                'hwid': self.hwid
            }, timeout=5)
            
            if response.status_code == 200:
                data = response.json()
                self.session_id = data.get('session_id')
        except:
            pass
    
    def heartbeat(self):
        """Envia sinal de vida para servidor"""
        try:
            requests.post(f"{API_URL}?action=heartbeat", data={
                'hwid': self.hwid,
                'session_id': self.session_id or 0
            }, timeout=5)
        except:
            pass
    
    def report_processes(self):
        """Envia lista de processos rodando"""
        try:
            # Pega todos os processos
            processes = []
            for proc in psutil.process_iter(['name']):
                try:
                    processes.append(proc.info['name'])
                except:
                    pass
            
            # Envia para servidor
            requests.post(f"{API_URL}?action=report_process_list", data={
                'hwid': self.hwid,
                'processes': json.dumps(processes)
            }, timeout=5)
        except:
            pass
    
    def report_hack_detection(self, process_name):
        """Reporta detecção de hack"""
        try:
            response = requests.post(f"{API_URL}?action=report_process", data={
                'hwid': self.hwid,
                'process_name': process_name
            }, timeout=5)
            
            if response.status_code == 200:
                data = response.json()
                if data.get('banned', False):
                    # Foi banido automaticamente
                    self.check_ban_status()
        except:
            pass
    
    def check_ban_status(self):
        """Verifica se foi banido"""
        try:
            response = requests.post(f"{API_URL}?action=check_ban", data={
                'hwid': self.hwid
            }, timeout=5)
            
            if response.status_code == 200:
                data = response.json()
                if data.get('banned', False):
                    self.banned.emit(data)
                    return True
        except:
            pass
        
        return False
    
    def stop(self):
        self.running = False


# ============================================
# INTEGRAÇÃO COM ANTIHACKMONITOR
# ============================================

class EnhancedAntiHackMonitor(QThread):
    """
    Versão melhorada do AntiHackMonitor com integração remota
    Substitua a classe AntiHackMonitor original por esta
    """
    hack_detected = pyqtSignal(str)
    pb_closed = pyqtSignal()
    
    def __init__(self, blacklist, remote_control):
        super().__init__()
        self.blacklist = blacklist
        self.remote_control = remote_control
        self.running = True
        self.pb_running = False
        
    def run(self):
        while self.running:
            try:
                # Verifica se Point Blank está rodando
                pb_found = False
                for proc in psutil.process_iter(['name']):
                    if proc.info['name'].lower() == 'pointblank.exe':
                        pb_found = True
                        self.pb_running = True
                        break
                
                if not pb_found and self.pb_running:
                    self.pb_running = False
                    self.pb_closed.emit()
                
                # Verifica processos da blacklist
                if self.pb_running:
                    for proc in psutil.process_iter(['name']):
                        proc_name = proc.info['name'].lower()
                        for blocked in self.blacklist:
                            if blocked.lower() in proc_name:
                                # Reporta para servidor ANTES de encerrar
                                self.remote_control.report_hack_detection(proc.info['name'])
                                
                                self.hack_detected.emit(proc.info['name'])
                                self.kill_pointblank()
                                return
                
                self.msleep(1000)
                
            except Exception as e:
                print(f"Erro no monitor: {e}")
                self.msleep(1000)
    
    def kill_pointblank(self):
        """Encerra o processo do Point Blank"""
        try:
            for proc in psutil.process_iter(['name']):
                if proc.info['name'].lower() == 'pointblank.exe':
                    proc.kill()
                    self.pb_running = False
        except Exception as e:
            print(f"Erro ao encerrar PB: {e}")
    
    def stop(self):
        self.running = False


# ============================================
# INSTRUÇÕES DE INTEGRAÇÃO
# ============================================
"""
COMO INTEGRAR NO LAUNCHER PRINCIPAL:

1. No __init__ do PBLauncher, adicione:
   
   self.hwid = get_hwid()
   self.remote_control = RemoteControl(self.hwid)
   self.remote_control.banned.connect(self.on_banned)
   self.remote_control.start()

2. Substitua a classe AntiHackMonitor por EnhancedAntiHackMonitor

3. No método start_monitor(), passe o remote_control:
   
   self.monitor = EnhancedAntiHackMonitor(self.blacklist, self.remote_control)

4. Adicione o método on_banned:
   
   def on_banned(self, ban_data):
       self.hide()
       
       msg = QMessageBox(self)
       msg.setIcon(QMessageBox.Critical)
       msg.setWindowTitle("Conta Bloqueada")
       msg.setText("Sua conta foi bloqueada!")
       
       reason = ban_data.get('reason', 'Violação dos termos de uso')
       permanent = ban_data.get('permanent', False)
       
       if permanent:
           detail = f"Motivo: {reason}\n\nBan PERMANENTE"
       else:
           until = ban_data.get('banned_until', 'Indefinido')
           detail = f"Motivo: {reason}\n\nBloqueado até: {until}"
       
       msg.setDetailedText(detail)
       
       # Botão de suporte
       support_btn = msg.addButton("Abrir Suporte", QMessageBox.ActionRole)
       msg.addButton(QMessageBox.Close)
       
       msg.exec_()
       
       if msg.clickedButton() == support_btn:
           import webbrowser
           webbrowser.open(ban_data.get('support_url', 'https://pbroyal.com/suporte'))
       
       QApplication.quit()

5. No método quit_application(), adicione:
   
   if self.remote_control and self.remote_control.isRunning():
       self.remote_control.stop()
       self.remote_control.wait()
"""
